'use client';

import Image from "next/image";
import { useLocale, useTranslations } from 'next-intl';
import { Link } from '@/i18n/routing';
import { usePathname } from 'next/navigation';
import { useState } from "react";

interface MenuItem {
  name: string;
  to: string;
  path: string;
  child?: MenuItem[];
}

export default function Navbar() {
  const locale = useLocale();
  const t = useTranslations('Navbar');
  const pathname = usePathname();
  const lastSegment = '/' + pathname.split('/').pop();

  const isLinkActive = (itemTo: string) => {
    const currentPath = pathname.replace(/^\/(en|ar)/, '') || '/';
    if (itemTo === '/' && currentPath === '/') return true;
    return itemTo !== '/' && currentPath.startsWith(itemTo);
  };

  const [openMenu, setOpenMenu] = useState<boolean>(false);
  const [activeSubMenu, setActiveSubMenu] = useState<number | null>(null);

  const menuItems: MenuItem[] = [
    { name: t("Home"), to: "/", path: "home" },
    { name: t("whoWeAre"), to: "/who-we-are", path: "who-we-are" },
    {
      name: t("aboutCompany"),
      to: "/about-company",
      path: "about-company",
      child: [
        { name: t("services"), to: "/services", path: "services" },
        { name: t("factory"), to: "/factory", path: "factory" },
        { name: t("equipment"), to: "/equipment", path: "equipment" },
        { name: t("certificates"), to: "/pages/certificates", path: "certificates" },
        { name: t("clients"), to: "/clients", path: "clients" },
        { name: t("downloads"), to: "/pages/downloads", path: "downloads" },
      ]
    },
    {
      name: t("projects"),
      to: "/projects",
      path: "projects",
      child: [
        { name: t("ourProjects"), to: "/pages/our-projects", path: "our-projects" },
        { name: t("ongoingProjects"), to: "/pages/ongoing-projects", path: "ongoing-projects" },
        { name: t("digitalGallery"), to: "/pages/digital-gallery", path: "digital-gallery" }
      ]
    },
    {
      name: t("community"),
      to: "/community",
      path: "community",
      child: [
        { name: t("news"), to: "/news", path: "news" },
        { name: t("careers"), to: "/careers", path: "careers" },
        { name: t("csr"), to: "/pages/csr", path: "csr" },
        { name: t("sustainability"), to: "/pages/sustainability", path: "sustainability" },
        { name: t("innovation"), to: "/pages/innovation", path: "innovation" },
      ]
    },
    { name: t("requestQuote"), to: "/request-quote", path: "request-quote" },
    { name: t("contactUs"), to: "/contact-us", path: "contact-us" },
  ];

  const handleMobileParentClick = (index: number, hasChildren: boolean) => {
    if (hasChildren) {
      setActiveSubMenu(activeSubMenu === index ? null : index);
    } else {
      setOpenMenu(false);
    }
  };

  return (
    <div className='flex flex-col w-full justify-center items-center h-22 bg-gray-100 shadow-sm relative z-50'>
      <div className='container flex w-full h-full flex-row justify-between items-center py-2 px-4'>

        <div className='flex w-fit justify-start'>
          <Link href={"/"} locale={locale}>
            <Image
              src="/assets/logos/logo.svg"
              alt='Logo'
              width={79 * 1.5}
              height={52 * 1.5}
              priority
            />
          </Link>
        </div>

        <div onClick={() => setOpenMenu(!openMenu)} className="relative group h-full flex md:hidden items-center cursor-pointer">
          <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="size-10 text-main-blue">
            <path strokeLinecap="round" strokeLinejoin="round" d="M3.75 6.75h16.5M3.75 12h16.5m-16.5 5.25h16.5" />
          </svg>
        </div>

        {/* Desktop Navigation */}
        <div className='hidden md:flex w-fit h-full items-center gap-6'>
          {menuItems.map((item, index) => {
            const hasChildren = item.child && item.child.length > 0;
            const isParentActive = isLinkActive(item.to);
            const isAnyChildActive = hasChildren ? item.child!.some(sub => isLinkActive(sub.to)) : false;
            const isActive = isParentActive || isAnyChildActive;

            return (
              <div key={index} className="relative group h-full flex items-center">
                <Link
                  href={item.to}
                  locale={locale}
                  className={`flex items-center gap-1 font-main text-[1rem] ${isActive ? 'text-main-orange' : 'text-main-blue'} hover:text-main-orange transition-colors duration-300 whitespace-nowrap`}
                >
                  {hasChildren && (
                    <svg className="w-4 h-4 transition-transform group-hover:rotate-180" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M19 9l-7 7-7-7" />
                    </svg>
                  )}
                  {item.name}
                </Link>

                {hasChildren && (
                  <div className="absolute top-full start-0 pt-2 opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-300 transform translate-y-2 group-hover:translate-y-0 z-50">
                    <div className="bg-white shadow-xl rounded-2xl p-3 min-w-75 flex flex-col gap-2 border border-gray-100">
                      {item.child!.map((subItem, subIndex) => (
                        <Link
                          key={subIndex}
                          href={subItem.to}
                          locale={locale}
                          className={`text-sm font-light px-3 py-2 rounded-lg transition-colors text-start ${isLinkActive(subItem.to) ? 'text-main-orange bg-orange-50' : 'text-gray-500 bg-gray-50 hover:text-main-orange'}`}
                        >
                          {subItem.name}
                        </Link>
                      ))}
                    </div>
                  </div>
                )}
              </div>
            );
          })}
        </div>

        {/* Mobile Fullscreen Menu */}
        {openMenu && (
          <div className="fixed inset-0 z-999999 w-full bg-white/60 backdrop-blur-xl drop-shadow-[0_4px_4px_rgba(0,0,0,0.05)] flex flex-col p-6 overflow-y-auto">
            <div className="flex justify-between items-center mb-10">
              <Image src="/assets/logos/logo.svg" alt='Logo' width={80} height={50} />
              <div onClick={() => { setOpenMenu(false); setActiveSubMenu(null); }} className="cursor-pointer">
                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className="size-10 text-main-blue">
                  <path strokeLinecap="round" strokeLinejoin="round" d="M6 18 18 6M6 6l12 12" />
                </svg>
              </div>
            </div>

            <div className="flex flex-col gap-4">
              {menuItems.map((item, index) => {
                const hasChildren = !!(item.child && item.child.length > 0);
                const isSubOpen = activeSubMenu === index;
                const isParentActive = isLinkActive(item.to) || (hasChildren && item.child!.some(sub => isLinkActive(sub.to)));

                return (
                  <div key={index} className="flex flex-col border-b border-gray-200 pb-3">
                    <div
                      className="flex justify-between items-center cursor-pointer"
                      onClick={() => handleMobileParentClick(index, hasChildren)}
                    >
                      <Link
                        href={hasChildren ? "#" : item.to}
                        locale={locale}
                        onClick={(e) => hasChildren && e.preventDefault()}
                        className={`text-lg font-medium ${isParentActive ? 'text-main-orange' : 'text-main-blue'}`}
                      >
                        {item.name}
                      </Link>

                      {hasChildren && (
                        <svg
                          className={`w-6 h-6 transition-transform duration-300 ${isSubOpen ? 'rotate-180 text-main-orange' : 'text-main-blue'}`}
                          fill="none" stroke="currentColor" viewBox="0 0 24 24"
                        >
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M19 9l-7 7-7-7" />
                        </svg>
                      )}
                    </div>

                    {hasChildren && isSubOpen && (
                      <div className="flex flex-col gap-3 mt-4 ml-4 pl-4 border-l-2 border-main-orange animate-in slide-in-from-top-1">
                        {item.child!.map((subItem, subIndex) => (
                          <Link
                            key={subIndex}
                            href={subItem.to}
                            locale={locale}
                            onClick={() => setOpenMenu(false)}
                            className={`text-md ${isLinkActive(subItem.to) ? 'text-main-orange font-bold' : 'text-gray-600'}`}
                          >
                            {subItem.name}
                          </Link>
                        ))}
                      </div>
                    )}
                  </div>
                );
              })}
            </div>
          </div>
        )}
      </div>
    </div>
  );
}