import type { Metadata } from "next";

interface MetadataProps {
  title: string;
  description: string;
  image?: string;
  noIndex?: boolean;
  keywords?: string[];
  canonicalUrlRelative?: string;
}

const siteConfig = {
  siteName: "شركة السلام للمقاولات",
  siteUrl: process.env.NEXT_PUBLIC_SITE_URL || "https://elsalam.net",
  defaultImage: "/assets/images/logos/logo.svg",
  twitterHandle: "@alsalam",
};

export function constructMetadata({
  title,
  description,
  image = siteConfig.defaultImage,
  noIndex = false,
  keywords = [],
  canonicalUrlRelative,
}: MetadataProps): Metadata {
  
  const fullUrl = canonicalUrlRelative 
    ? `${siteConfig.siteUrl}${canonicalUrlRelative}` 
    : siteConfig.siteUrl;

  return {
    // 1. الأساسيات
    title: {
      default: `${title} | ${siteConfig.siteName}`,
      template: `%s | ${siteConfig.siteName}`, // هيظهر: من نحن | شركة السلام للمقاولات
    },
    description: description,
    keywords: keywords,
    applicationName: siteConfig.siteName,
    metadataBase: new URL(siteConfig.siteUrl),

    // 2. إعدادات الزحف (Robots)
    robots: {
      index: !noIndex,
      follow: !noIndex,
      googleBot: {
        index: !noIndex,
        follow: !noIndex,
      },
    },

    // 3. Open Graph (Facebook, LinkedIn, WhatsApp)
    openGraph: {
      title: title,
      description: description,
      url: fullUrl,
      siteName: siteConfig.siteName,
      images: [
        {
          url: image,
          width: 1200,
          height: 630,
          alt: title,
        },
      ],
      locale: "ar_EG",
      type: "website",
    },

    // 4. Twitter
    twitter: {
      card: "summary_large_image",
      title: title,
      description: description,
      images: [image],
      creator: siteConfig.twitterHandle,
    },

    // 5. Canonical URL
    alternates: {
      canonical: fullUrl,
    },
    
    icons: {
      icon: '/favicon.ico',
      shortcut: '/favicon-16x16.png',
      apple: '/apple-touch-icon.png',
    },
  };
}